# Copyright (C) 2018 Jiaan Dai

"""Script for generating configs containing random testing data points.

To run the generated testing data points, please use `python
scripts\run_test.py config_filename` in the package root.
"""

import argparse
import random
import json

CONFIG_TEMPLATE = """{{
    "target_database": "uniprot-all.fasta",
    "decoy_database": "uniprot-homo-sapiens.shuffle.fasta",
    "peptide_mass_range": [700, 5000],
    "ms1tol": 250,
    "ms2tol": 0.01,
    "max_mass_shift": 250,

    "missing_proportions": {missing},
    "signal_levels": {noise},
    "num_mods": {mod},

    "tag_length": 3,
    "size": 40000,

    "output_path": "{path}",
    "num_threads": 24,
    "seed": {seed}
}}
"""


def main(rs_from, rs_to, logrn_from, logrn_to, lambda_from, lambda_to,
         num_points, result_name, config_path):
    mu_list = []
    rho_list = []
    num_mods_list = []

    for i in range(0, num_points):
        mu = random.uniform(1 - rs_to, 1 - rs_from)
        rho_exponent = random.uniform(-logrn_to, -logrn_from)
        rho = 10 ** rho_exponent
        num_mods = random.uniform(lambda_from, lambda_to)

        mu_list.append(mu)
        rho_list.append(rho)
        num_mods_list.append(num_mods)

    json_data = CONFIG_TEMPLATE.format(missing=json.dumps(mu_list),
                                       noise=json.dumps(rho_list),
                                       mod=json.dumps(num_mods_list),
                                       path=result_name,
                                       seed=random.randint(1000, 8192))

    with open(config_path, 'w') as file:
        file.write(json_data)


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('--rs_from', required=True, type=float,
                        help='`r_s` range starting point')
    parser.add_argument('--rs_to', required=True, type=float,
                        help='`r_s` range ending point')
    parser.add_argument('--logrn_from', required=True, type=float,
                        help='`log10 r_n` range starting point')
    parser.add_argument('--logrn_to', required=True, type=float,
                        help='`log10 r_n` range ending point')
    parser.add_argument('--lambda_from', required=True, type=float,
                        help=r'`\lambda` range starting point')
    parser.add_argument('--lambda_to', required=True, type=float,
                        help=r'`\lambda` range ending point')
    parser.add_argument('--num', required=True, type=int,
                        help='data points number')
    parser.add_argument('--result', required=True, help='result name')
    parser.add_argument('-o', required=True, help='output config path')
    args = parser.parse_args()
    random.seed()
    main(args.rs_from, args.rs_to, args.logrn_from, args.logrn_to,
         args.lambda_from, args.lambda_to, args.num, args.result, args.o)
