# Copyright (C) 2017 Jiaan Dai

"""Record aggregator for simulation results.

RecordWriter exports newline-delimited raw json data to the disk. RecordMerger
collects the raw jsons from the disk, then merge them into the target format.
The first line is global information. Then each block starts with a local params
json, following by local_params['size'] json record. Before writing into the
disk, RecordMerger will do a gz compression.
"""

import os
import json
import gzip
from datetime import datetime as _datetime


class RecordWriter:
    """Writer used in the individual simulator."""

    def __init__(self, filename):
        self._filename = filename
        self._buffer = []

    def append(self, record):
        line = json.dumps(record)
        self._buffer.append(line)

    def write(self, local_params):
        with open(self._filename + '.dat', 'w') as f:
            f.write(local_params)
            f.write('\n')
            for line in self._buffer:
                f.write(line)
                f.write('\n')


class RecordMerger:
    """Merger to aggregate results in individual simulators."""

    def __init__(self, outputname, description, params, filenames):
        self._outputname = outputname
        self._description = description
        self._filenames = filenames
        self._params = params

    def run(self):
        header = {
            'description': self._description,
            'version': _datetime.now().strftime('%Y-%m-%d_%H-%M-%S'),
            'global_params': self._params
        }

        with gzip.open(self._outputname + '.dat', 'wb') as f:
            f.write(bytes(json.dumps(header), encoding='utf-8'))
            f.write(b'\n')
            for rawfile in self._filenames:
                with open(rawfile + '.dat', 'r') as rf:
                    f.write(bytes(rf.read(), encoding='utf-8'))

        for rawfile in self._filenames:
            os.remove(rawfile + '.dat')
